﻿using System;
using System.ComponentModel;
using System.Data;
using System.Data.Common;
using Microsoft.Data.Tools.Schema.Sql.UnitTesting;
using Microsoft.Data.Tools.Schema.Sql.UnitTesting.Conditions;

namespace ColumnCountCondition
{
  [System.ComponentModel.DesignerCategory("Code")]
  [ExportTestCondition("Column Count", typeof(ColumnCountCondition))]
  public class ColumnCountCondition : TestCondition
  {
    private int _resultSet;
    private int _count;
    private int _batch;

    public ColumnCountCondition()
    {
      _resultSet = 1;
      _count = 0;
      _batch = 1;
    }

    //method you need to override  
    //to perform the condition verification  
    public override void Assert(DbConnection validationConnection, SqlExecutionResult[] results)
    {
      //call base for parameter validation  
      base.Assert(validationConnection, results);

      //verify batch exists  
      if (results.Length < _batch)
        throw new DataException(String.Format("Batch {0} does not exist", _batch));

      SqlExecutionResult result = results[_batch - 1];

      //verify resultset exists  
      if (result.DataSet.Tables.Count < ResultSet)
        throw new DataException(String.Format("ResultSet {0} does not exist", ResultSet));

      DataTable table = result.DataSet.Tables[ResultSet - 1];

      //actual condition verification  
      //verify resultset column count matches expected  
      if (table.Columns.Count != Count)
        throw new DataException(String.Format(
            "ResultSet {0}: {1} columns did not match the {2} columns expected",
            ResultSet, table.Columns.Count, Count));
    }

    //this method is called to provide the string shown in the  
    //test conditions panel grid describing what the condition tests  
    public override string ToString()
    {
      return String.Format(
          "Condition fails if ResultSet {0} does not contain {1} columns",
          ResultSet, Count);
    }

    //below are the test condition properties  
    //that are exposed to the user in the property browser  
    #region Properties  

    //property specifying the resultset for which  
    //you want to check the column count  
    [Category("Test Condition")]
    [DisplayName("ResultSet")]
    [Description("ResultSet Number")]
    public int ResultSet
    {
      get { return _resultSet; }

      set
      {
        //basic validation  
        if (value < 1)
          throw new ArgumentException("ResultSet cannot be less than 1");

        _resultSet = value;
      }
    }

    //property specifying  
    //expected column count  
    [Category("Test Condition")]
    [DisplayName("Count")]
    [Description("Column Count")]
    public int Count
    {
      get { return _count; }

      set
      {
        //basic validation  
        if (value < 0)
          throw new ArgumentException("Count cannot be less than 0");

        _count = value;
      }
    }

    #endregion
  }
}